﻿param
(
    [Parameter(Position=0,Mandatory=$true)]
    [System.String]$DeviceId,
    [Parameter(Position=1,Mandatory=$true)]
    [System.String]$DesiredPortName
)

#region Functions

Function Enable-Privilege
{
  param(
    ## The privilege to adjust. This set is taken from
    ## http://msdn.microsoft.com/en-us/library/bb530716(VS.85).aspx
    [ValidateSet(
      "SeAssignPrimaryTokenPrivilege", "SeAuditPrivilege", "SeBackupPrivilege",
      "SeChangeNotifyPrivilege", "SeCreateGlobalPrivilege", "SeCreatePagefilePrivilege",
      "SeCreatePermanentPrivilege", "SeCreateSymbolicLinkPrivilege", "SeCreateTokenPrivilege",
      "SeDebugPrivilege", "SeEnableDelegationPrivilege", "SeImpersonatePrivilege", "SeIncreaseBasePriorityPrivilege",
      "SeIncreaseQuotaPrivilege", "SeIncreaseWorkingSetPrivilege", "SeLoadDriverPrivilege",
      "SeLockMemoryPrivilege", "SeMachineAccountPrivilege", "SeManageVolumePrivilege",
      "SeProfileSingleProcessPrivilege", "SeRelabelPrivilege", "SeRemoteShutdownPrivilege",
      "SeRestorePrivilege", "SeSecurityPrivilege", "SeShutdownPrivilege", "SeSyncAgentPrivilege",
      "SeSystemEnvironmentPrivilege", "SeSystemProfilePrivilege", "SeSystemtimePrivilege",
      "SeTakeOwnershipPrivilege", "SeTcbPrivilege", "SeTimeZonePrivilege", "SeTrustedCredManAccessPrivilege",
      "SeUndockPrivilege", "SeUnsolicitedInputPrivilege")]
    $Privilege,
    ## The process on which to adjust the privilege. Defaults to the current process.
    $ProcessId = $pid,
    ## Switch to disable the privilege, rather than enable it.
    [Switch] $Disable
  )

  ## Taken from P/Invoke.NET with minor adjustments.
  $definition = @'
  using System;
  using System.Runtime.InteropServices;
   
  public class AdjPriv
  {
    [DllImport("advapi32.dll", ExactSpelling = true, SetLastError = true)]
    internal static extern bool AdjustTokenPrivileges(IntPtr htok, bool disall,
      ref TokPriv1Luid newst, int len, IntPtr prev, IntPtr relen);
   
    [DllImport("advapi32.dll", ExactSpelling = true, SetLastError = true)]
    internal static extern bool OpenProcessToken(IntPtr h, int acc, ref IntPtr phtok);
    [DllImport("advapi32.dll", SetLastError = true)]
    internal static extern bool LookupPrivilegeValue(string host, string name, ref long pluid);
    [StructLayout(LayoutKind.Sequential, Pack = 1)]
    internal struct TokPriv1Luid
    {
      public int Count;
      public long Luid;
      public int Attr;
    }
   
    internal const int SE_PRIVILEGE_ENABLED = 0x00000002;
    internal const int SE_PRIVILEGE_DISABLED = 0x00000000;
    internal const int TOKEN_QUERY = 0x00000008;
    internal const int TOKEN_ADJUST_PRIVILEGES = 0x00000020;
    public static bool EnablePrivilege(long processHandle, string privilege, bool disable)
    {
      bool retVal;
      TokPriv1Luid tp;
      IntPtr hproc = new IntPtr(processHandle);
      IntPtr htok = IntPtr.Zero;
      retVal = OpenProcessToken(hproc, TOKEN_ADJUST_PRIVILEGES | TOKEN_QUERY, ref htok);
      tp.Count = 1;
      tp.Luid = 0;
      if(disable)
      {
        tp.Attr = SE_PRIVILEGE_DISABLED;
      }
      else
      {
        tp.Attr = SE_PRIVILEGE_ENABLED;
      }
      retVal = LookupPrivilegeValue(null, privilege, ref tp.Luid);
      retVal = AdjustTokenPrivileges(htok, false, ref tp, 0, IntPtr.Zero, IntPtr.Zero);
      return retVal;
    }
  }
'@

  $processHandle = (Get-Process -id $ProcessId).Handle
  $type = Add-Type $definition -PassThru
  $success=$type[0]::EnablePrivilege($processHandle, $Privilege, $Disable)
  return $success
}

Function Get-SerialPorts
{
    $serialDevices=@{}
    $portDevices=Get-WmiObject -Namespace ROOT\WMI -Class MSSerial_PortName
    foreach ($portDevice in $portDevices)
    {
        $hwid=$portDevice.InstanceName
        $hwid=$hwid.Remove(($hwid.Length-2),2)
        $serialDevices.Add($hwid,$portDevice.PortName)
    }
    $serialPorts=Get-WmiObject -Class Win32_SerialPort
    foreach ($serialPort in $serialPorts)
    {
        if(-not $serialDevices.ContainsKey($serialPort.PNPDeviceId))
        {
            $serialDevices.Add($serialPort.PNPDeviceId,$serialPort.DeviceId)
        }
    }
    return $serialDevices
}

Function Switch-SerialPort
{
    param
    (
        [Parameter(Position=0,Mandatory=$true)]
        [System.String]$PortName,
        [Parameter(Position=1,Mandatory=$true)]
        [System.String]$NewPortName,
        [Parameter(Position=2,Mandatory=$false)]
        [System.String]$ReassignmentStart="COM5"
    )
    
    $aclInherit=[System.Security.AccessControl.InheritanceFlags]::ContainerInherit
    $aclPropagation=[System.Security.AccessControl.PropagationFlags]::None
    
    $TargetDeviceKey=""
    $ExistingDeviceKey=""
    $serialPorts=Get-SerialPorts
    
    #We may need to push our reassignment up...
    if($NewPortName -eq $ReassignmentStart)
    {
        $i=([System.Convert]::ToInt32($ReassignmentStart.Replace("COM","")))+1
        $ReassignmentStart="COM$i"
    }

    #Find the device we want to setup and any one which may be in it's place
    foreach ($serialPort in $serialPorts.Keys)
    {
        $deviceId=$serialPort
        $port=$serialPorts[$serialPort]
        $portFromRegistry=Get-PortFromRegistry -PnpDeviceId $deviceId
        #Did this already get reset to another port???
        if($portFromRegistry -ne $port)
        {
            $port=$portFromRegistry
        }
        if($PortName -eq $port)
        {
            Write-Host "Found target match with device $deviceId`n"
            $TargetDeviceKey="HKLM:\System\CurrentControlSet\Enum\$deviceId"
        }
        elseif($NewPortName -eq $port)
        {
            Write-Host "Found existing match with device $deviceId`n"
            $ExistingDeviceKey="HKLM:\System\CurrentControlSet\Enum\$deviceId"
        }
        elseif($ReassignmentStart -eq $port)
        {
            Write-Host "A device is already present at $ReassignmentStart`n"
            $i=([System.Convert]::ToInt32($ReassignmentStart.Replace("COM","")))+1
            $ReassignmentStart="COM$i"
            Write-Host "Port Reassignment will now start at $ReassignmentStart"
        }
    }

    if(-not [System.String]::IsNullOrEmpty($TargetDeviceKey))
    {
        $ntAccount=New-Object System.Security.Principal.NTAccount([System.Security.Principal.WindowsIdentity]::GetCurrent().Name)
        $newRule=New-Object System.Security.AccessControl.RegistryAccessRule("Administrators","FullControl",$aclInherit,$aclPropagation,"Allow")
        
        if(-not [System.String]::IsNullOrEmpty($ExistingDeviceKey))
        {
            Write-Host "An existing device has port $NewPortName`n"
            $keyname=$ExistingDeviceKey.Replace("HKLM:\","")
            if(Enable-Privilege SeTakeOwnershipPrivilege)
            {
                $key = [Microsoft.Win32.Registry]::LocalMachine.OpenSubKey($keyname,[Microsoft.Win32.RegistryKeyPermissionCheck]::ReadWriteSubTree,[System.Security.AccessControl.RegistryRights]::TakeOwnership)
                $acl=$key.GetAccessControl()
                $acl.SetOwner($ntAccount)        
                $key.SetAccessControl($acl)
                $key.Close()

                $key = [Microsoft.Win32.Registry]::LocalMachine.OpenSubKey($keyname,[Microsoft.Win32.RegistryKeyPermissionCheck]::ReadWriteSubTree,[System.Security.AccessControl.RegistryRights]::TakeOwnership)
                $acl=$key.GetAccessControl()
                $acl.SetAccessRule($newRule)
                $key.SetAccessControl($acl)
                $key.Close()

                #We'll now change the Friendly Name
                $friendlyName=(Get-ItemProperty -Path $ExistingDeviceKey).FriendlyName
                $friendlyName=$friendlyName.Replace($NewPortName,$ReassignmentStart)
                Set-ItemProperty -Path $ExistingDeviceKey -Name "FriendlyName" -Value $friendlyName -Force
            
                #Now change the Port Name
                Set-ItemProperty -Path "$ExistingDeviceKey\Device Parameters" -Name "PortName" -Value $ReassignmentStart
    
                #Set the ACL back the way it was...
                $key = [Microsoft.Win32.Registry]::LocalMachine.OpenSubKey($keyname,[Microsoft.Win32.RegistryKeyPermissionCheck]::ReadWriteSubTree,[System.Security.AccessControl.RegistryRights]::TakeOwnership)
                $acl=$key.GetAccessControl()
                $acl.RemoveAccessRule($newRule)|Out-Null
                $key.SetAccessControl($acl)
                $key.Close()
        
                Write-Host "Switched $NewPortName to $ReassignmentStart`n"
            }
            else
            {
                Write-Host "Unable to set SeTakeOwnershipPrivilege on Token`n"
            }
        }

        #We need to adjust the ACL
        $keyname=$TargetDeviceKey.Replace("HKLM:\","")
        if(Enable-Privilege SeTakeOwnershipPrivilege)
        {
            $key = [Microsoft.Win32.Registry]::LocalMachine.OpenSubKey($keyname,[Microsoft.Win32.RegistryKeyPermissionCheck]::ReadWriteSubTree,[System.Security.AccessControl.RegistryRights]::TakeOwnership)
            $acl=$key.GetAccessControl()
            $acl.SetOwner($ntAccount)        
            $key.SetAccessControl($acl)
            $key.Close()

            $key = [Microsoft.Win32.Registry]::LocalMachine.OpenSubKey($keyname,[Microsoft.Win32.RegistryKeyPermissionCheck]::ReadWriteSubTree,[System.Security.AccessControl.RegistryRights]::TakeOwnership)
            $acl=$key.GetAccessControl()
            $acl.SetAccessRule($newRule)
            $key.SetAccessControl($acl)
            $key.Close()

            #We'll now change the Friendly Name
            $friendlyName=(Get-ItemProperty -Path $TargetDeviceKey).FriendlyName
            $friendlyName=$friendlyName.Replace($PortName,$NewPortName)
            Set-ItemProperty -Path $TargetDeviceKey -Name "FriendlyName" -Value $friendlyName -Force
            
            #Now change the Port Name
            Set-ItemProperty -Path "$TargetDeviceKey\Device Parameters" -Name "PortName" -Value $NewPortName
    
            #Set the ACL back the way it was...
            $key = [Microsoft.Win32.Registry]::LocalMachine.OpenSubKey($keyname,[Microsoft.Win32.RegistryKeyPermissionCheck]::ReadWriteSubTree,[System.Security.AccessControl.RegistryRights]::TakeOwnership)
            $acl=$key.GetAccessControl()
            $acl.RemoveAccessRule($newRule)|Out-Null
            $key.SetAccessControl($acl)
            $key.Close()
        
            Write-Host "Switched $PortName to $NewPortName`n"
        }
        else
        {
            Write-Host "Unable to set SeTakeOwnershipPrivilege on Token`n"
        }
    }
    else
    {
        Write-Host "There is not a device present with $PortName`n"
    }

}

Function Get-PortFromRegistry
{
    param
    (
        [System.String]$PnpDeviceId
    )
    $portname=""
    $regkey="HKLM:\System\CurrentControlSet\Enum\$PnpDeviceId\Device Parameters"
    $regval=Get-ItemProperty -Path $regkey -ErrorAction SilentlyContinue
    if($regval -ne $null)
    {
        $portname=$regval.PortName
    }
    return $portname
}

#endregion


Start-Transcript -Path "$env:TEMP\COMShuffling.log" -Append

Write-Host "Searching for $DeviceId To Set Desired Port:$DesiredPortName`n"

#Find the device..
$ports=Get-SerialPorts
foreach ($port in $ports.Keys)
{
    if($port -like "$DeviceId*")
    {
        Write-Host "Match Found for $DeviceId on $($ports[$port])!`n"
        Switch-SerialPort -PortName $ports[$port] -NewPortName $DesiredPortName
    }
}

Stop-Transcript|Out-Null

